﻿namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Hims.Shared.UserModels;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using System;
   using System.Collections.Generic;
   using System.Linq;
   using System.Threading.Tasks;

    /// <inheritdoc />
    [Authorize]
    [Route("api/appointment-types")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class AppointmentTypesController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IAppointmentTypesService appointmentTypesService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public AppointmentTypesController(IAppointmentTypesService appointmentTypesService, IAuditLogService auditLogServices)
        {
            this.appointmentTypesService = appointmentTypesService;
            this.auditLogServices = auditLogServices;
        }


        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] AppointmentTypesModel model, [FromHeader] LocationHeader header)
        {
            model = (AppointmentTypesModel)EmptyFilter.Handler(model);
            var response = await this.appointmentTypesService.InsertAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.AppointmentTypes,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" <b>{model.CreatedByName}</b> has <b>added</b> Appointment Type <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAllAsync([FromBody] AppointmentTypesModel model)
        {
            model ??= new AppointmentTypesModel();
            var response = await this.appointmentTypesService.FetchAllAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateHocAsync([FromBody] AppointmentTypesModel model, [FromHeader] LocationHeader header)
        {
            model = (AppointmentTypesModel)EmptyFilter.Handler(model);
            var response = await this.appointmentTypesService.UpdateAsync(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.AppointmentTypes,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" <b>{model.CreatedByName}</b> has <b>updated</b> Appointment Type <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }
        /// <summary>
        /// Deletes the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("delete")]
        public async Task<ActionResult> DeleteAsync([FromBody] AppointmentTypesModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (AppointmentTypesModel)EmptyFilter.Handler(model);
                var response = await this.appointmentTypesService.DeleteAsync(model);
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.AppointmentTypes,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" <b>{model.CreatedByName}</b> has <b>deleted</b> Appointment Type <b>{model.Name}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success(response);
            }
            catch (Exception)
            {
                return this.Success(-3);
            }
        }

    }
}
